<?php

namespace App\Http\Services;

use App\Models\MappingGuru;
use App\Models\MappingSiswa;
use App\Models\NilaiKeterampilan;
use App\Models\NilaiPas;
use App\Models\Mapel;
use App\Models\NilaiPengetahuan;
use App\Models\NilaiPts;
use App\Models\RateNilai;
use App\Models\Semester;
use App\Models\TahunAjaran;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class NilaiService
{
    private int $tahunAjaranId;
    private MappingSiswa $mappingSiswa;
    private $rasio;
    private $semester;

    public function __construct(MappingSiswa $mappingSiswa)
    {
        $activeTahunAjaran = TahunAjaran::where('is_active', true)->first();
        $rasio = RateNilai::where('is_active', true)->first();
        $activeSemester = Semester::where('is_active', true)->first();

        $this->mappingSiswa = $mappingSiswa;
        $this->tahunAjaranId = $activeTahunAjaran->id;
        $this->rasio = $rasio;
        $this->semester = $activeSemester->id;
    }

    public function calculate(string $group, string $tingkat): Collection|array
    {
        $mapelNilaiPengetahuan = DB::table('mapels')
            ->select('mapels.kelompok', 'mapels.nama_mapel', 'mapping_guru.kelas_id', 'mapping_guru.guru_id', 'mapping_guru.id AS mapping_guru_id', 'nilai_pengetahuan.siswa_id', 'nilai_pengetahuan.nilai', 'mapping_guru.tahun_ajaran_id', 'nilai_pengetahuan.semester')
            ->leftJoin('mapping_guru', 'mapels.id', '=', 'mapping_guru.mapel_id')
            ->leftJoin('nilai_pengetahuan', function ($join) {
                $join->on('mapping_guru.id', '=', 'nilai_pengetahuan.mapping_guru_id')
                    ->where('nilai_pengetahuan.siswa_id', '=', $this->mappingSiswa->siswa_id)
                    ->where('nilai_pengetahuan.semester', $this->semester)
                    ->where('nilai_pengetahuan.tahun_ajaran_id', $this->tahunAjaranId);
            })
            ->where('kelompok', $group)
            ->where('tingkat', $tingkat)
            ->orderBy('nama_mapel','ASC')
            ->get();

        $mapelNilaiKeterampilan = DB::table('mapels')
            ->select('mapels.kelompok', 'mapels.nama_mapel', 'mapping_guru.kelas_id', 'mapping_guru.guru_id', 'mapping_guru.id AS mapping_guru_id', 'nilai_keterampilan.siswa_id', 'nilai_keterampilan.nilai', 'mapping_guru.tahun_ajaran_id', 'nilai_keterampilan.semester')
            ->leftJoin('mapping_guru', 'mapels.id', '=', 'mapping_guru.mapel_id')
            ->leftJoin('nilai_keterampilan', function ($join) {
                $join->on('mapping_guru.id', '=', 'nilai_keterampilan.mapping_guru_id')
                    ->where('nilai_keterampilan.siswa_id', '=', $this->mappingSiswa->siswa_id)
                    ->where('nilai_keterampilan.semester', $this->semester)
                    ->where('nilai_keterampilan.tahun_ajaran_id', $this->tahunAjaranId);
            })
            ->where('kelompok', $group)
            ->where('tingkat', $tingkat)
            ->orderBy('nama_mapel','ASC')
            ->get();

        $nilaiA = $mapelNilaiPengetahuan->groupBy('nama_mapel');
        $nilaiKA = $mapelNilaiKeterampilan->groupBy('nama_mapel');

        // Calculate Pengetahuan
        foreach ($nilaiA as $namaMapel => $nilaiMapel) {
            foreach ($nilaiMapel as $nm) {
                if ($nm->kelas_id == $this->mappingSiswa->kelas_id && $nm->tahun_ajaran_id == $this->tahunAjaranId) {
                    $mapelId = MappingGuru::find($nm->mapping_guru_id)->mapel->id;
                    $getMappingGuru = MappingGuru::where('tahun_ajaran_id', $this->tahunAjaranId)
                        ->where('guru_id', $nm->guru_id)
                        ->where('kelas_id', $nm->kelas_id)
                        ->where('mapel_id', $mapelId)
                        ->first();

                    $getNilaiPts = $this->getNilaiPts($getMappingGuru->id);
                    $getNilaiPas = $this->getNilaiPas($getMappingGuru->id);

                    $getKdPengetahuan = NilaiPengetahuan::where('siswa_id', $this->mappingSiswa->siswa_id)
                        ->where('tahun_ajaran_id', $this->tahunAjaranId)
                        ->where('semester', $this->semester)
                        ->where('mapping_guru_id', $getMappingGuru->id)
                        ->get();
                }
            }

            $nilaiPas = $getNilaiPas ?? 0;
            $nilaiPengetahuan = $nilaiMapel->avg('nilai') ?? 0;
            $nilaiPts = $getNilaiPts ?? 0;

            $nilaiAkhirPengetahuan = $this->calculateNilaiAkhir($nilaiPengetahuan, $nilaiPts, $nilaiPas);

            $nilaiA[$namaMapel]['rataRataPengetahuan'] = ($nilaiAkhirPengetahuan !== 0) ? $nilaiAkhirPengetahuan : 0;
            if (isset($getKdPengetahuan)) {
                $nilaiA[$namaMapel]['kd_pengetahuan'] = $this->stringifyKdName($getKdPengetahuan);
            }
        }

        // Calculate Keterampilan
        foreach ($nilaiKA as $namaMapel => $nilaiMapel) {
            foreach ($nilaiMapel as $nm) {
                if ($nm->kelas_id == $this->mappingSiswa->kelas_id && $nm->tahun_ajaran_id == $this->tahunAjaranId) {
                    $mapelId = MappingGuru::find($nm->mapping_guru_id)->mapel->id;
                    $getMappingGuru = MappingGuru::where('tahun_ajaran_id', $this->tahunAjaranId)
                        ->where('guru_id', $nm->guru_id)
                        ->where('kelas_id', $nm->kelas_id)
                        ->where('mapel_id', $mapelId)
                        ->first();
                    $nilaiPts = $this->getNilaiPts($getMappingGuru->id);
                    $nilaiPas = $this->getNilaiPas($getMappingGuru->id);

                    $getKdKeterampilan = NilaiKeterampilan::where('siswa_id', $this->mappingSiswa->siswa_id)
                        ->where('tahun_ajaran_id', $this->tahunAjaranId)
                        ->where('semester', $this->semester)
                        ->where('mapping_guru_id', $getMappingGuru->id)
                        ->get();
                }
            }

            // Menghitung rata-rata nilai pengetahuan dan keterampilan
            $nilaiKeterampilan = $nilaiMapel->avg('nilai') ?? 0;

            $nilaiAkhirKeterampilan = $this->calculateNilaiAkhir($nilaiKeterampilan, $nilaiPts, $nilaiPas);

            $nilaiA[$namaMapel]['rataRataKeterampilan'] = ($nilaiAkhirKeterampilan !== 0) ? $nilaiAkhirKeterampilan : 0;
            $nilaiA[$namaMapel]['rataRata'] = (int) number_format(($nilaiA[$namaMapel]['rataRataPengetahuan'] + $nilaiA[$namaMapel]['rataRataKeterampilan']) / 2);
            if (isset($getKdKeterampilan)) {
                $nilaiA[$namaMapel]['kd_keterampilan'] = $this->stringifyKdName($getKdKeterampilan);
            }
        }

        return [
            'listOfMapel' => $mapelNilaiPengetahuan,
            'calculatedNilai' => $nilaiA
        ];
    }



    public function getNilaiPts(int $mappingGuruId): int
    {
        try {
            $queryNilai = NilaiPts::where('siswa_id', $this->mappingSiswa->siswa_id)
                ->where('tahun_ajaran_id', $this->tahunAjaranId)
                ->where('semester', $this->semester)
                ->where('mapping_guru_id', $mappingGuruId)
                ->first();
            $nilai = $queryNilai->nilai;
        } catch (\Exception $e) {
            $nilai = 0;
//            Log::info($e->getMessage());
        }
        return $nilai;
    }

    public function getNilaiPas(int $mappingGuruId): int
    {
        try {
            $queryNilai = NilaiPas::where('siswa_id', $this->mappingSiswa->siswa_id)
                ->where('tahun_ajaran_id', $this->tahunAjaranId)
                ->where('semester', $this->semester)
                ->where('mapping_guru_id', $mappingGuruId)
                ->first();
            $nilai = $queryNilai->nilai;
        } catch (\Exception $e) {
            $nilai = 0;
//            Log::info($e->getMessage());
        }
        return $nilai;
    }

    public function calculateNilaiAkhir(float $nilaiHarian, float $nilaiPts, float $nilaiPas): int
    {
        $countRasioNilaiPengetahuan = ($nilaiHarian * $this->rasio->rate_nilai_harian) / 100;
        $countRasioNilaiPts = ($nilaiPts * $this->rasio->rate_pts) / 100;
        $countRasioNilaiPas = ($nilaiPas * $this->rasio->rate_pas) / 100;

        return number_format($countRasioNilaiPengetahuan, 0) + number_format($countRasioNilaiPts, 0) + number_format($countRasioNilaiPas, 0);
    }

    public function stringifyKdName(Collection $allKd): string
    {
        $tempKd = [];
        foreach ($allKd as $keyKd) {
            $tempKd[] = $keyKd->kd->nama_kd;
        }
        return implode('; ', $tempKd);
    }
}
